from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import os
import logging

import torch
import torch.nn as nn
import torch.nn.functional as F
from collections import OrderedDict

from .unet_aps.apspool import ApsDown, get_pad_layer, Down2X
from lib.models.unet_aps.unet_aps_parts import UpforResnet

BN_MOMENTUM = 0.1
logger = logging.getLogger(__name__)


def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes, momentum=BN_MOMENTUM)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes, momentum=BN_MOMENTUM)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, cfg, inplanes, planes, stride=1, downsample=None, downsample_conv=None, downsample_bn=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes, momentum=BN_MOMENTUM)

        self.filter_size = cfg.MODEL.EXTRA.FILTER_SIZE
        self.padding_mode = cfg.MODEL.EXTRA.PADDING_MODE
        self.aps_criterion = cfg.MODEL.EXTRA.APS_CRITERION
        self.downsample_type = cfg.MODEL.EXTRA.DOWNSAMPLE_TYPE
        self.is_poly = (self.downsample_type == "aps" or self.downsample_type == "random")

        self.stride = stride

        if self.stride == 2:
            self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1,
                          padding=1, padding_mode=self.padding_mode, bias=False)
            self.conv2_down = Down2X(downsample_type=self.downsample_type)
        else:
            self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1,
                          padding=1, padding_mode=self.padding_mode, bias=False)

        self.bn2 = nn.BatchNorm2d(planes, momentum=BN_MOMENTUM)
        self.conv3 = nn.Conv2d(planes, planes * self.expansion, kernel_size=1,
                               bias=False)
        self.bn3 = nn.BatchNorm2d(planes * self.expansion,
                                  momentum=BN_MOMENTUM)
        self.relu = nn.ReLU(inplace=True)

        self.downsample_conv = downsample_conv
        self.downsample = downsample
        self.downsample_bn = downsample_bn

    def forward(self, input):

        if isinstance(input, tuple):
            x = input[0]
            polyphase_comp = input[1]
        else:
            x = input
            polyphase_comp = None

        residual = x
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        if self.stride == 2 and self.is_poly:
            out = self.conv2(out)
            out, polyphase_comp = self.conv2_down(out)
        elif self.stride == 2:
            out = self.conv2(out)
            out = self.conv2_down(out)
        elif self.stride == 1:
            out = self.conv2(out)

        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            if self.stride == 2:
                residual = self.downsample_conv(x)

                if self.is_poly:
                    residual, r_ = self.downsample({'inp': residual, 'polyphase_indices': polyphase_comp})
                    assert 0 == ((r_ != polyphase_comp).sum()), 'ERROR: downsample grid not equal'
                else:
                    residual = self.downsample(residual)

                residual = self.downsample_bn(residual)
            else:
                residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        if self.is_poly:
            return out, polyphase_comp
        else:
            return out


class PoseResNet(nn.Module):
    def __init__(self, block, layers, cfg, **kwargs):
        self.inplanes = 64
        self.cfg = cfg
        extra = cfg.MODEL.EXTRA
        self.deconv_with_bias = extra.DECONV_WITH_BIAS
        super(PoseResNet, self).__init__()

        self.filter_size = extra.FILTER_SIZE
        self.padding_mode = extra.PADDING_MODE
        self.aps_criterion = extra.APS_CRITERION
        self.downsample_type = extra.DOWNSAMPLE_TYPE
        self.upsample_type = extra.UPSAMPLE_TYPE
        self.is_poly = (self.downsample_type == "aps" or self.downsample_type == "random")

        self.conv1 = nn.Sequential(nn.Conv2d(3, 64, kernel_size=7, stride=1, padding=3, padding_mode=self.padding_mode,
                                             bias=False),
                                   Down2X(downsample_type=self.downsample_type)
                                   )
        self.bn1 = nn.BatchNorm2d(64, momentum=BN_MOMENTUM)
        self.relu = nn.ReLU(inplace=True)

        self.maxpool = nn.Sequential(
            get_pad_layer(self.padding_mode)((1, 1, 1, 1)),
            nn.MaxPool2d(kernel_size=3, stride=1),
            Down2X(downsample_type=self.downsample_type)
        )

        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2)

        if self.upsample_type == "aps" or self.upsample_type == "aps_conv":
            self.deconv_layers1 = UpforResnet(512 * 4, extra.NUM_DECONV_FILTERS[0], bilinear=False,
                                              padding_mode=self.padding_mode,
                                              filter_size=self.filter_size)
            self.deconv_layers2 = UpforResnet(extra.NUM_DECONV_FILTERS[0], extra.NUM_DECONV_FILTERS[1], bilinear=False,
                                              padding_mode=self.padding_mode,
                                              filter_size=self.filter_size)
            self.deconv_layers3 = UpforResnet(extra.NUM_DECONV_FILTERS[1], extra.NUM_DECONV_FILTERS[2], bilinear=False,
                                              padding_mode=self.padding_mode,
                                              filter_size=self.filter_size)
            self.deconv_layers4 = UpforResnet(cfg.MODEL.NUM_JOINTS, cfg.MODEL.NUM_JOINTS,
                                              bilinear=(self.upsample_type != "aps_conv"),
                                              padding_mode=self.padding_mode,
                                              filter_size=self.filter_size)
            self.deconv_layers5 = UpforResnet(cfg.MODEL.NUM_JOINTS, cfg.MODEL.NUM_JOINTS,
                                              bilinear=(self.upsample_type != "aps_conv"),
                                              padding_mode=self.padding_mode,
                                              filter_size=self.filter_size)
        else:
            self.deconv_layers = self._make_deconv_layer(
                extra.NUM_DECONV_LAYERS,
                extra.NUM_DECONV_FILTERS,
                extra.NUM_DECONV_KERNELS,
            )

        self.final_layer = nn.Conv2d(
            in_channels=extra.NUM_DECONV_FILTERS[-1],
            out_channels=cfg.MODEL.NUM_JOINTS,
            kernel_size=extra.FINAL_CONV_KERNEL,
            stride=1,
            padding=1 if extra.FINAL_CONV_KERNEL == 3 else 0,
            padding_mode=self.padding_mode
        )

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        downsample_conv = None
        downsample_bn = None

        if stride != 1 or self.inplanes != planes * block.expansion:
            if stride == 1:
                downsample = nn.Sequential(
                    nn.Conv2d(self.inplanes, planes * block.expansion,
                              kernel_size=1, stride=1, bias=False),
                    nn.BatchNorm2d(planes * block.expansion, momentum=BN_MOMENTUM),
                )
            else:
                downsample_conv = nn.Conv2d(self.inplanes, planes * block.expansion,
                                            kernel_size=1, stride=1, bias=False)
                downsample = Down2X(downsample_type=self.downsample_type)
                downsample_bn = nn.BatchNorm2d(planes * block.expansion, momentum=BN_MOMENTUM)

        layers = [block(self.cfg, self.inplanes, planes, stride, downsample, downsample_conv, downsample_bn)]
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.cfg, self.inplanes, planes))

        return nn.Sequential(*layers)

    def _get_deconv_cfg(self, deconv_kernel, index):
        if deconv_kernel == 4:
            padding = 1
            output_padding = 0
        elif deconv_kernel == 3:
            padding = 1
            output_padding = 1
        elif deconv_kernel == 2:
            padding = 0
            output_padding = 0

        return deconv_kernel, padding, output_padding

    def _make_deconv_layer(self, num_layers, num_filters, num_kernels):
        assert num_layers == len(num_filters), \
            'ERROR: num_deconv_layers is different len(num_deconv_filters)'
        assert num_layers == len(num_kernels), \
            'ERROR: num_deconv_layers is different len(num_deconv_filters)'

        layers = []
        for i in range(num_layers):
            kernel, padding, output_padding = \
                self._get_deconv_cfg(num_kernels[i], i)

            planes = num_filters[i]
            layers.append(
                nn.ConvTranspose2d(
                    in_channels=self.inplanes,
                    out_channels=planes,
                    kernel_size=kernel,
                    stride=2,
                    padding=padding,
                    output_padding=output_padding,
                    bias=self.deconv_with_bias))
            layers.append(nn.BatchNorm2d(planes, momentum=BN_MOMENTUM))
            layers.append(nn.ReLU(inplace=True))
            self.inplanes = planes

        return nn.Sequential(*layers)

    def forward(self, x):
        if self.is_poly:
            x, polyphase_comp1 = self.conv1(x)
            x = self.bn1(x)
            x = self.relu(x)
            x, polyphase_comp2 = self.maxpool(x)
            # backbone
            x = self.layer1(x)
            x, polyphase_comp3 = self.layer2(x)
            x, polyphase_comp4 = self.layer3(x)
            x, polyphase_comp5 = self.layer4(x)
        else:
            x = self.conv1(x)
            x = self.bn1(x)
            x = self.relu(x)
            x = self.maxpool(x)
            # backbone
            x = self.layer1(x)
            x = self.layer2(x)
            x = self.layer3(x)
            x = self.layer4(x)

        # deconv
        if self.upsample_type == "aps" or self.upsample_type == "aps_conv":
            x = self.deconv_layers1(x, polyphase_comp5)
            x = self.deconv_layers2(x, polyphase_comp4)
            x = self.deconv_layers3(x, polyphase_comp3)
            x = self.final_layer(x)
            x = self.deconv_layers4(x, polyphase_comp2)
            x = self.deconv_layers5(x, polyphase_comp1)[:, :, ::4, ::4]
        else:
            x = self.deconv_layers(x)
            x = self.final_layer(x)

        return x

    def init_weights(self, pretrained=''):
        if os.path.isfile(pretrained):
            if self.upsample_type == "aps" or self.upsample_type == "aps_conv":
                logger.info('=> init deconv weights from normal distribution')
                for name, m in self.deconv_layers1.named_modules():
                    if isinstance(m, nn.Conv2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
                for name, m in self.deconv_layers2.named_modules():
                    if isinstance(m, nn.Conv2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
                for name, m in self.deconv_layers3.named_modules():
                    if isinstance(m, nn.Conv2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
                for name, m in self.deconv_layers4.named_modules():
                    if isinstance(m, nn.Conv2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
                for name, m in self.deconv_layers5.named_modules():
                    if isinstance(m, nn.Conv2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
            else:
                logger.info('=> init deconv weights from normal distribution')
                for name, m in self.deconv_layers.named_modules():
                    if isinstance(m, nn.ConvTranspose2d):
                        logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.normal_(m.weight, std=0.001)
                        if self.deconv_with_bias:
                            nn.init.constant_(m.bias, 0)
                    elif isinstance(m, nn.BatchNorm2d):
                        logger.info('=> init {}.weight as 1'.format(name))
                        logger.info('=> init {}.bias as 0'.format(name))
                        nn.init.constant_(m.weight, 1)
                        nn.init.constant_(m.bias, 0)
            logger.info('=> init final conv weights from normal distribution')
            for m in self.final_layer.modules():
                if isinstance(m, nn.Conv2d):
                    # nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                    # logger.info('=> init {}.weight as normal(0, 0.001)'.format(name))
                    # logger.info('=> init {}.bias as 0'.format(name))
                    nn.init.normal_(m.weight, std=0.001)
                    nn.init.constant_(m.bias, 0)

            # pretrained_state_dict = torch.load(pretrained)
            logger.info('=> loading pretrained model {}'.format(pretrained))
            # self.load_state_dict(pretrained_state_dict, strict=False)
            checkpoint = torch.load(pretrained)
            if isinstance(checkpoint, OrderedDict):
                state_dict = checkpoint
            elif isinstance(checkpoint, dict) and 'state_dict' in checkpoint:
                state_dict_old = checkpoint['state_dict']
                state_dict = OrderedDict()
                # delete 'module.' because it is saved from DataParallel module
                for key in state_dict_old.keys():
                    if key.startswith('module.'):
                        # state_dict[key[7:]] = state_dict[key]
                        # state_dict.pop(key)
                        state_dict[key[7:]] = state_dict_old[key]
                    else:
                        state_dict[key] = state_dict_old[key]
            else:
                raise RuntimeError(
                    'No state_dict found in checkpoint file {}'.format(pretrained))
            self.load_state_dict(state_dict, strict=False)
        else:
            logger.error('=> imagenet pretrained model dose not exist')
            logger.error('=> please download it first')
            raise ValueError('imagenet pretrained model does not exist')


resnet_spec = {18: (BasicBlock, [2, 2, 2, 2]),
               34: (BasicBlock, [3, 4, 6, 3]),
               50: (Bottleneck, [3, 4, 6, 3]),
               101: (Bottleneck, [3, 4, 23, 3]),
               152: (Bottleneck, [3, 8, 36, 3])}


def get_pose_net(cfg, is_train, **kwargs):
    num_layers = cfg.MODEL.EXTRA.NUM_LAYERS
    style = cfg.MODEL.STYLE

    block_class, layers = resnet_spec[num_layers]

    print('use ASYM Bottleneck_APS! Layers:%s' % num_layers)

    if style == 'antialiased':
        raise Exception('Implementation with Bottleneck_antialiased mode currently not supported.')

    model = PoseResNet(block_class, layers, cfg, **kwargs)

    if is_train and cfg.MODEL.INIT_WEIGHTS:
        model.init_weights(cfg.MODEL.PRETRAINED)

    return model
